<?php

/** 
 * This module contains all funtions used by all admin modules. I.e:
 * 
 *  admin.php, add_gallery.php, modify_gallery.php, delete_gallery.php
 * 
 * @package admin
 */


/**
 * load_gd tries to load gd library in case it is not automatically loaded.
 * Then checks the availability of some functions included in this library.
 *
 * Note: this function does not check if gd library is 2 or higher.
 * 
 * @param bool $debug if true shows some debug messages. Used in installation.
 * @return bool true if library loaded, false if not.
 */

function load_gd($debug=false){

  if (!extension_loaded('gd')) {
    if ($debug) warning(_y("GD library not loaded. Will try to load it."));
    if (!function_exists('dl')){
      if ($debug) 
	warning(_y("Function dynamic extension load not available."));
      return(false);
    }

    if (@dl('php_gd2.dll')|| @dl('php_gd.dll') ){
      if ($debug) msg(_y("php_gd(2).dll sucessfully loaded."));
    }
    else {
      if(!@dl('gd.so'))	{
	if ($debug) warning(_y("GD Library NOT AVAILABLE"));
	return(false);
      }
      if ($debug) msg(_y("gd.so sucessfully loaded."));
    }
  }
  
  //these functions are needed.
  if (!function_exists('getimagesize')) { 
    if ($debug) warning(_y("Function getimagesize does not exist"));
      return(false);
  }
  if (!function_exists('imagecreatetruecolor')&& 
      !function_exists('imagecreate')){
    if ($debug) warning(_y("Function imagecreate(truecolor) does not exist"));
    return(false);
  }
  if (!function_exists('imagecreatefromjpeg')) {
    if ($debug) warning(_y("Function imacreatefromjpeg does not exist"));
    return(false);
  }
  if ($debug) msg(_y("GD library is loaded."));
  return(true);
}


/**
 * Prints admin login form. Uses login_form.
 * 
 * @return none
 * @global template directory
 * 
 */


function login_form() {
    global $TEMPLATE_DIR;
    
    //Admin login form internationalization.
    $I_LOGIN=_y('Please login as admin');
    $I_USER=_y('User');
    $I_PASSWORD=_y('Password');
    $I_SUBMIT=_y('Submit');
    include($TEMPLATE_DIR . 'login_form.php');
}

/**
 * Admin task bar contains all possible task the YaPiG admin can do.
 * This function prints this task bar. Uses admin_task_bar.php
 * 
 * @global template location
 */

function print_admin_taskbar() {
  global $TEMPLATE_DIR;
  //I18n 
  $I_ADMIN_TASKS=_y("Administration tasks");
  $I_UPLOAD=_y('Upload files');
  $I_ADD=_y('Add gallery');
  $I_MODIFY=_y('Modify gallery');
  $I_DELETE=_y('Delete gallery');
  $I_CHANGE=_y('Change main config');
  $I_LOGOUT=_y('Logout');
  include ($TEMPLATE_DIR . 'admin_task_bar.php');
  
}

/*
 * Prints add/modify gallery information form. Used for adding a gallery
 * and for modifying gallery information such as title, author, description..
 * 
 * @param string $D_ACTION destination url of the action attribute
 * @param string $D_TITLE gallery title
 * @param string $D_AUTHOR gallery author
 * @param string $D_DIR gallery dir
 * @param string $D_DESCRIPTION gallery description
 * @param string $D_PASSWORD gallery password
 * @global galleries base directory
 * @global directory where template is located
 */

function print_gallery_form($D_ACTION, $gallery_info=array()) {
    global $BASE_DIR,$TEMPLATE_DIR;

    $D_TITLE=$gallery_info['title'];
    $D_AUTHOR=$gallery_info['author'];
    $D_DESCRIPTION=$gallery_info['desc'];
    $D_PASSWORD=$gallery_info['gallery_password'];
    $D_DATE=$gallery_info['date'];
    if ($gallery_info['no_comments']=="on") {
      $D_CHECKED="checked=\"checked\"";
    }


    $I_GALLERY_TITLE=_y('Gallery title');
    $I_AUTHOR = _y('Author');
    $I_DATE= _y('Date');
    $I_DATE_WARNING=_y("Free Format. Leave it blank to set today's date.");
    $I_LOCATION = _y('Location');
    $I_PASSWORD = _y('Password');
    $I_PASS_WARNING= _y('set this field if you want private access to this gallery');
    $I_DESCRIPTION = _y('Description');
    $I_NO_COMMENTS=_y("Hide comments");
    $I_SEND=_y('Send');
    
    $D_BASE_DIR=$BASE_DIR;


    include($TEMPLATE_DIR . "gallery_form.php");
}

/////////// ACTIONS WITH GALLERIES ///////////////////



/**
 * Creates a thumbnail image from a jpeg file.
 *
 * It takes as arguments the paths to the source image and the path/name of 
 * the destination image. Uses the global variables to set the destination
 * image size.
 * 
 * @param string $path_dest Destination Path + name of output file
 * @param string $path_origen Source image path. Includes de name of the image
 * 
 * @global max size in pixels
 * @global jpeg quality of thumbs
 * 
 * @returns bool false if there was any problem :)
 * 
 */

function create_thumb ($path_dest,$path_origen) {
    global $THUMB_SIZE, $THUMB_QUALITY;
    
    echo  "<div> $path_origen";
    if (!file_exists($path_origen)) return(false);
    
    if (!($sz_orig = getimagesize ($path_origen))) return(false);
    

    $img_pixels=$sz_orig[0] * $sz_orig[1];
    if ($img_pixels>3900000){
	warning( $path_origen . _y("  is a very big image, it might cause memory problems."));
    }
    /*
     * Cut from getimagesize() documentation:
     * Returns an array with 4 elements. Index 0 contains the width of the
     * image in pixels. Index 1 contains the height. Index 2 is a flag 
     * indicating the type of the image: 1 = GIF, 2 = JPG, 3 = PNG, 4 = SWF,
     * 5 = PSD, 6 = BMP, 7 = TIFF(intel byte order), 8 = TIFF(motorola 
     * byte order), 9 = JPC, 10 = JP2, 11 = JPX, 12 = JB2, 13 = SWC, 
     * 14 = IFF. These values correspond to the IMAGETYPE constants that 
     * were added in PHP 4.3.
     */
    switch($sz_orig[2]) {
     case 1:
	$im_orig=imagecreatefromgif ($path_origen);
	break;
     case 2:	
	$im_orig=imagecreatefromjpeg ($path_origen);
	break;
     case 3:
       warning(_y("PNG files not supported in this version"));
       return(true);
       //$im_orig=imagecreatefrompng ($path_origen);      
       
	break;
     default: return(false);
    }
    //Now We calculate the thumbnail size. 
    $ratio=$sz_orig[1]/$sz_orig[0];
    if ($ratio>1) {
	$height=$THUMB_SIZE;
	$width=(int)($THUMB_SIZE/$ratio);
    }
    else {
	$width=$THUMB_SIZE;
	$height=(int) ($THUMB_SIZE*$ratio);
    }
    if (function_exists('imagecreatetruecolor')) {// GD v2.0
      if(!($im_dest = imagecreatetruecolor ($width,$height)))
	return(false);
    }
    else{
      if(!($im_dest = imagecreate ($width,$height)))
	return(false);
    }
    

    
    echo " ({$sz_orig[0]}x{$sz_orig[1]}) -> <b>$path_dest ({$width}x{$height}))</b></div>";
    if(!imagecopyresampled($im_dest,$im_orig,0,0,0,0,$width,$height,
			   $sz_orig[0],$sz_orig[1])) return(false);
    if (!imagejpeg($im_dest, $path_dest, $THUMB_QUALITY)) return(false);
    return(true);
}

/**
 * This function creates all thumbnails 
 * 
 * files are stored in src_dir and will be stored in dest_dir prefixing $THUMB_PREFIX
 * to de image name.
 * 
 * @param string $dest_dir  Path to destination dir
 * @param string $src_dir   Path to source dir
 * @param array $array_filenames  Contains the names of the source images (ex: file.jpg) 
 * 
 * @global string $THUMB_PREFIX
 * @return bool
 */

function create_all_thumbs ($dest_dir,$src_dir,$array_filenames) {
    global $THUMB_PREFIX;

    //First of all check if gd library is already loaded
     if(!load_gd(true)) {
      error(_y("GD library not loaded. Please read install documentation"));
     }
    
    
    foreach($array_filenames as $src_file) {
	$src_path= $src_dir . $src_file;
	$dest_path= $dest_dir . $THUMB_PREFIX . $src_file;  
	if (!create_thumb($dest_path,$src_path)) return(false);
    }
    return(true);
}

/**
 * Gets all image filenames of the dir
 * 
 * This function scans all filenames that are in a directory and
 * returns the names of the image files that are cotained in.
 * 
 * @param string $dir path of the dir.
 * 
 * @global string preceding the name of the thumbnail
 * @global array with the valid extensions of the images
 * @return array With the image filenames that the directory contains.
 */

function get_all_image_filenames ($dir) {
    global $THUMB_PREFIX,$IMAGE_EXT;
    
    $list=array();
    if (!is_dir($dir)) return(false);
    if (!($dh = opendir($dir))) return(false);
    while (($file = readdir($dh)) !== false) {
	if (is_file($dir . $file)) {
	    $fileinfo = pathinfo($file);
	    if(in_array(strtolower($fileinfo["extension"]),$IMAGE_EXT)){
		if(!preg_match("/$THUMB_PREFIX/",$fileinfo['basename'])){
		  array_push($list,$file);
		}
	    }
	}
    }
    closedir($dh);
    return($list);	    
}

/**
 * creates phid filenames correspondence file with this line format:
 *  phid $EQUAL filename\n
 *  
 * @param string $gid_dir directory where images are stored (with $BASE_DIR)
 * @param array $filenames array with this format: array[phid]=filename.
 * @global name of the created file
 * @global field separator
 * @return bool true if ok, false if not.
 * 
 */

function create_phid_filenames_file($gid_dir,$filenames) {	    
    global $PHID_FILENAMES, $EQUAL;
    
    //If not is array => create file with nothing
    if (!is_array($filenames)) {
      if (!($fd=fopen($gid_dir . $PHID_FILENAMES,"w+"))) return(false);
      fclose($fd);
      return(true);
    }
    $i=0;
    if (!($fd=fopen($gid_dir . $PHID_FILENAMES,"w+"))) return(false);
    foreach($filenames as $key => $fname) {
	fputs($fd,$key . $EQUAL . $fname . "\n");
    }
    fclose($fd);
    return(true);
}

/**
 * Creates file witch contains information about the gallery.Called
 * $GID_INFO_FILE witch is a php file that contains
 *
 * $new array has these keys:
 * 
 * - root_gid: not used
 * - title: gallery title
 * - author: author of the gallery.
 * - gallery_password: password of the gallery.
 * - thumb_phid : thumbnail phid.
 * - no_comments: do not allow, comments (1|0)
 * - num_images : number of images
 * - num_subgids: number of sub galleries. Not used
 * - date: date of creation
 * - desc: lines describing the contents of the gallery.
 *
 * if file does not exist => all keys need to be set.
 * @param string $gallery_path path to gallery, includes $BASE_DIR
 * @param array $new array with gallery information
 * @return bool true if ok, false if error :D
*/
function create_info_file($gallery_path,$new,$update=true){
    global $GID_INFO_FILE, $TEMPLATE_DIR;
    
    //gif_path=gid_info_file path
    $gif_path=$gallery_path . $GID_INFO_FILE;

    //If file exists => only update values included in new)
    //If it not an update => overwrite file.
    if (!file_exists($gif_path)||(!$update)){
      $info=$new;
    }
    else { 
	include($gif_path);  
	foreach ($gid_info as $key => $value) {
	  //echo "{gid_info[$key]}=> $value; new[{$new[$key]}] [$set] <br />\n";
	  if (isset($new[$key]))  $info[$key]=$new[$key];
	  else 
	    $info[$key]=$gid_info[$key];
	}//end foreach
    }
    
    //echo "<pre>";print_r($info);echo"</pre>";
    if (!($fd=fopen($gallery_path . $GID_INFO_FILE,"w+"))) return(false);
    fputs($fd,"<?php\n");
    fputs($fd,"\$gid_info['root_gid']=\"" . $info['root_gid']. "\";\n");
    fputs($fd,"\$gid_info['title']=\"". str_replace("\\'","'",$info['title']). "\";\n");
    fputs($fd,"\$gid_info['author']=\"". str_replace("\\'","'",$info['author']). "\";\n");
    fputs($fd,"\$gid_info['gallery_password']=\"".
	       $info['gallery_password']. "\";\n");
    fputs($fd,"\$gid_info['thumb_phid']=\"" . $info['thumb_phid'] . "\";\n");
    fputs($fd,"\$gid_info['no_comments']=\"" . $info['no_comments'] . "\";\n");
    fputs($fd,"\$gid_info['num_images']=\"". $info['num_images']. "\";\n");
    fputs($fd,"\$gid_info['num_subgids']=\"". $info['num_subgids']. "\";\n");
    fputs($fd,"\$gid_info['date']=\"". $info['date']. "\";\n");
    fputs($fd,"\$gid_info['desc']=\"". str_replace("\\'","'",$info['desc']). "\";\n");
    fputs($fd,"?>");
    fclose($fd);
    return(true);
}

/**
 * Gets the value of the gid counter and increases it one unit.
 * @global counter file witch is incremented each time this func is called
 * @return integer valid gallery identificator., that is,one not being used.
 */

function get_gid_counter() {
    global  $GID_COUNTER_FILE;
    
    if (!file_exists($GID_COUNTER_FILE)){
	if (!($fd=fopen($GID_COUNTER_FILE,'w'))){
	    error(_y("Opening gid counter file: " . $GID_COUNTER_FILE));
	}
	fputs($fd,'1');
	fclose($fd);
	return(1);
    }
    if (!($fd=fopen($GID_COUNTER_FILE,'r+'))){
	error(_y("Opening gid counter file: " . $GID_COUNTER_FILE));
    }
    $counter=fgets($fd,10);
    $counter++;
    rewind($fd);
    fputs($fd,$counter);
    fclose($fd);
    return($counter);
      
}
/**
 * Checks if dir is being used by other gallery.
 * 
 * Searchs in $GID_DIR file if $dir is being used by another
 * gallery. 
 * 
 * @param string $dir to check. Ended in '/'. Without $BASE_DIR
 * @global gid dirs correspondence file
 * @global field separator
 * @return bool true if dir is being used. false if not.
 */
function gid_dir_being_used($dir) {
 global $GID_DIRS, $EQUAL;
    
    //If it is first gallery => not being used
    if(!file_exists($GID_DIRS)) return(false);
    if(filesize($GID_DIRS)==0) return(false);
    //If could not load => Print warning.
    if (!($tmp=get_all_data($GID_DIRS))) {
	warning("gid_dir_being_used: Check file permissions of: " . $GID_DIR);
	return(false);
    }
    if (in_array($dir, $tmp)) return(true);
    return(false);
}

/**
 * Adds $dir to file $GID_DIRS, witch contains gid->dir correspondence
 * 
 * @param string $dir dir name without $BASE_DIR
 * @global gid->dirs correspondence filename.
 * @global gid->dirs field separator
 * @return bool true if ok, false if error.
 */

function add_to_gid_dirs($dir) {
    global $GID_DIRS, $EQUAL;
    
    //Load file
    $tmp=array();
    if (file_exists($GID_DIRS)) $tmp=file($GID_DIRS);
    //Update file
    if (!($fd=fopen($GID_DIRS,"w+"))) return(false);
    fputs($fd, get_gid_counter() . $EQUAL . $dir . "\n");
    foreach ($tmp as $line) 
      fputs($fd,$line);
    fclose($fd);
    return(true);
}

/**
 * shows modify gid options (links)
 * 
 * @param string $gid_dir gallery directory with $BASE_DIR 
 * @param integer $gid gallery identificator
 *
 * @global file with info about the contents of the gallery
 * @global template dir
 * @global phid comments stats file.
 * @global gid stats file.
 * @global total counter stats identifier.
 * @return bool true if ok false if not.
 */

function print_modify_gid($gid_dir,$gid){
  global $GID_INFO_FILE,$TEMPLATE_DIR,$PHID_COMMENTS,$GID_STATS,$TOTAL_LIST;

  if (!file_exists($gid_dir . $GID_INFO_FILE)) return(false);
  
  include ($gid_dir . $GID_INFO_FILE);
  
  $D_VISITS=get_data($GID_STATS,$gid);
  $D_COMMENTS=get_data($gid_dir. $PHID_COMMENTS, $TOTAL_LIST);
  $D_TITLE=$gid_info['title'];
  $D_AUTHOR=$gid_info['author'];
  $D_NUMBER_IMAGES=$gid_info['num_images'];
  $D_DATE=$gid_info['date'];
  $D_DESC=$gid_info['desc'];
  $D_THUMB=get_thumb_path($gid_dir, $gid_info['thumb_phid'],true);
  $I_MOD_INFO=_y('Modify gallery info');
  $I_SEL_THUMB=_y('Select thumbnail');
  $I_MOD_PHIDS=_y('Modify image list');
  $I_DEL_STUFF=_y('Delete gallery');
  $D_GID=$gid;
  $I_NUMBER_IMAGES=_y('Number of images');
  $I_TITLE=_y('Title');
  $I_AUTHOR=_y('Author');
  $I_VIEW_GALLERY=_y('View this gallery');
  $I_GALLERY=_y('Gallery');
  $I_VISITS=_y('Visits');
  $I_COMMENTS=_y('Comments');
  $I_DATE=_y('Date');
  $I_DESCRIPTION=_y('Description');
  
  include($TEMPLATE_DIR . 'modify_gid.php');
  return(true);
}


/**
 * Shows form when admin wants to delete a gallery.
 * 
 * Uses 'delete_gallery_form.php'
 *  
 * @param integer $D_GID gallery identificator
 * @global template directory
 *
 */
function print_delete_gallery_form($D_GID){
  global $TEMPLATE_DIR;

  $I_SELECT_OPT=_y('Select deletion options:');
  $I_NOT_LIST=_y('Do not list gallery in gallery index any more.');
  $I_DEL_COMMENTS=_y('Clear all image comments.');
  $I_DEL_IMG_COUNTERS=_y('Clear all image visit counters.');
  $I_DEL_GAL_COUNTERS=_y('Clear gallery visit counters.');
  $I_DEL_THUMBS=_y('Delete gallery thumbnails.');
  $I_DEL_ALL_FILES=_y('Delete all files in directory.');
  $I_SEND=_y('send');

  include($TEMPLATE_DIR . 'delete_gallery_form.php');
}

/**
 * Obtains the images from a list that do not have a thumbnail already 
 * created
 *
 * @param array $all_list list with filenames
 * @param string $gid_dir is the directory where images are stored
 * @return mixed false if problem. array with images without thumb if ok.
 */

function get_images_without_thumb($gid_dir,$all_list) {
  global $THUMB_PREFIX;

  if (!is_array($all_list)) return(false);
  foreach ($all_list as $one_image) {
    if (!@file_exists($gid_dir . $THUMB_PREFIX . $one_image)) {
      $return_list[]=$one_image;
    }
  }
  if (!is_array($return_list)) return(false);
  return($return_list);
  
}



?>
